/******************************************************************************
    SG_LED_PWM.c      Set RGB colour and brightness at a finger touch!
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This draws 3 bargraphs on the GLCD for Red Green Blue and the user can
  touch the 3 bargraphs to set the brightness of the backlight LEDs
  using PWM. The PWM is handled invisibly in a simple TMR0 interrupt that
  does the PWM at about 280 Hz which is fast enough that it doesn't cause
  flickering against the LCD pixel refresh rate.
******************************************************************************/
// Global Variables
unsigned char tp_x;     // holds results of Touch Panel adc values
unsigned char tp_y;
unsigned char tp_sx;    // holds Touch Panel screen pixel X and Y values
unsigned char tp_sy;
unsigned int tp_math;   // used for 16bit TP coords calcs

// defines used for the bargraphs
#define BAR_X       80    // XY is bar top left corner
#define BAR_LENGTH  151
#define BAR_Y       20
#define BAR_HEIGHT  14
unsigned char boff;       // bar Y offset used to draw multiple bars
unsigned char barsize;    // used for drawing bargraph, size of bar
unsigned char i;

// defines used for the buttons
#define BUTY    92
#define BUTY2  106

// defines used for the LED PWM system
unsigned char int_count;
unsigned char pwm_red;    // pwm values for display
unsigned char pwm_green;
unsigned char pwm_blue;
unsigned char pwm_red_mem;    // memorised pwm values
unsigned char pwm_green_mem;
unsigned char pwm_blue_mem;

unsigned char text[4];  // holds 3 digit number + NULL, for text display

// TPfast definitions
void TPfast_SetX(void);   // function to set X axis for next read
void TPfast_SetY(void);   // function to set X axis for next read
void TPfast_StartRead(void);      // function to start adc read from TouchPanel
unsigned char TPfast_Read(void);  // function to read adc result from TouchPanel

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

#include "TPfast.c"     // my fast TouchPanel functions to read X and Y

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void interrupt()
{
  //-------------------------------------------------------
  // TMR0 overflow interrupt; prescale 1:1
  // The TMR0 int performs PWM on the backlight RGB LED.
  // This is a fast smiple int, basically it just compares the int_count
  // to the PWM value for each colour.
  // TMR0L bit7 is forced hi, so int happens every 128+2 ticks
  // (total PWM cycle is about 240 Hz)
  //-------------------------------------------------------
  if(!int_count)    // when int_count==0, turn off all LEDs
  {
    BacklightRed   = 1;    // 1 is off 
    BacklightGreen = 1;  
    BacklightBlue  = 1;  
  }
  else       // for 1-255, some LEDs might be on
  {
    // turn each LED on when reaches pwm value
    if(int_count == pwm_red)   BacklightRed   = 0;
    if(int_count == pwm_green) BacklightGreen = 0;
    if(int_count == pwm_blue)  BacklightBlue  = 0;
  }
  int_count--;          // inc sequence (must count DOWN)
  TMR0L.F7 = 1;         // make TMR0 roll over every 128+2 ticks
  INTCON.TMR0IF = 0;    // clear the int flag before exit
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//=============================================================================
//  BLANK
//=============================================================================
void blank(void)
{
  //-------------------------------------------------------

}
//-----------------------------------------------------------------------------

//=============================================================================
//  DRAW PWM VALUE
//=============================================================================
void Draw_PWM_Value(unsigned char pc)
{
  //-------------------------------------------------------
  // draws the Red Green Blue pwm values as text to the screen
  //-------------------------------------------------------
  if(pc == 0)
  {
    ByteToStr(pwm_red,text);
    T6963C_Write_Text(text, 6, 3, T6963C_ROM_MODE_OR);
  }
  if(pc == 1)
  {
    ByteToStr(pwm_green,text);
    T6963C_Write_Text(text, 6, 6, T6963C_ROM_MODE_OR);
  }
  if(pc == 2)
  { 
    ByteToStr(pwm_blue,text);
    T6963C_Write_Text(text, 6, 9, T6963C_ROM_MODE_OR);
  }
}
//-----------------------------------------------------------------------------

//=============================================================================
//   DRAW BARGRAPH
//=============================================================================
void Draw_Bargraph(unsigned char bc)
{
  //-------------------------------------------------------
  // crude function, draws 3 bargraphs for; Red Green Blue
  //-------------------------------------------------------
  // get pam and y drawing offset for the selected bargraph
  boff = 0;
  tp_math = pwm_red;
  if(bc == 1)
  {
    boff = 24;
    tp_math = pwm_green;  
  }
  if(bc == 2)
  {
    boff = 24+24;
    tp_math = pwm_blue;  
  }

  // convert pwm value 0-255 to a bar size value 0-BAR_LENGTH
  tp_math = (tp_math * BAR_LENGTH);
  tp_math = (tp_math / 255);
  barsize = tp_math;        // is now range 0-BAR_LENGTH

  // draw outer bar frame
  T6963C_rectangle(BAR_X-2, BAR_Y-2+boff, (BAR_X+BAR_LENGTH)+1, (BAR_Y+BAR_HEIGHT)+1+boff, T6963C_WHITE);  
  
  // loop and draw bar size contents
  for(i=0; i<=(BAR_LENGTH-3); i+=4)
  {
    if(i < barsize) T6963C_Rectangle(BAR_X+i,BAR_Y+boff,BAR_X+i+2,(BAR_Y+BAR_HEIGHT-1)+boff,T6963C_WHITE);
    else            T6963C_Rectangle(BAR_X+i,BAR_Y+boff,BAR_X+i+2,(BAR_Y+BAR_HEIGHT-1)+boff,T6963C_BLACK);
  }
}
//-----------------------------------------------------------------------------

//=============================================================================
//  DRAW BUTTON
//=============================================================================
void Draw_Button(unsigned char but, unsigned char highlight)
{
  //-------------------------------------------------------
  // crude function to draw either of 2 buttons to GLCD
  //-------------------------------------------------------
  if(but == 1)
  {
    if(highlight) T6963C_box(131+1,BUTY+1,131+76,BUTY2-1,T6963C_WHITE); // filled
    else          T6963C_box(131+1,BUTY+1,131+76,BUTY2-1,T6963C_BLACK); // plain
    T6963C_rectangle(131,BUTY,131+76,BUTY2,T6963C_WHITE);         // outline
    T6963C_Write_Text("Memorise",  17, 12, T6963C_ROM_MODE_XOR);  // text
  }
  else
  {
    if(highlight) T6963C_box(35+1,BUTY+1,35+76,BUTY2-1,T6963C_WHITE); // filled
    else          T6963C_box(35+1,BUTY+1,35+76,BUTY2-1,T6963C_BLACK); // plain
    T6963C_rectangle(35,BUTY,35+76,BUTY2,T6963C_WHITE);         // outline
    T6963C_Write_Text("Playback",  5, 12, T6963C_ROM_MODE_XOR); // text
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  TP CONVERT XY
//=============================================================================
void TP_convert_XY(void)
{
  //-------------------------------------------------------
  // the TouchPanel ADC XY values (0-255) are already in tp_x and tp_y
  // this converts them to a calibrated screen pixel X Y from
  // 0-239 and 0-127 and stores them in variables tp_sx and tp_sy.
  // This is optimised for execution speed so you need to enter
  // in constants for your TouchPanel XY ADC values.
  // NOTE!! tp_math must be a 16bit variable; unsigned int tp_math
  // Conversion is done by simple math; Xpixels = (Xadc * 240 / Xadc_range)
  //-------------------------------------------------------
  #define TP_CAL_XMIN   12    // enter ADC values for your screen pixel edges
  #define TP_CAL_XMAX   241   // these worked good on my GLCD
  #define TP_CAL_YMIN   21
  #define TP_CAL_YMAX   227
  #define TP_CAL_XRANGE (TP_CAL_XMAX - TP_CAL_XMIN)   // don't edit these 2
  #define TP_CAL_YRANGE (TP_CAL_YMAX - TP_CAL_YMIN)

  // Convert X adc value to screen pixels
  if(tp_x > TP_CAL_XMIN)            // make sure is >0 before doing calc!
  {
    tp_math = (tp_x - TP_CAL_XMIN);     // get X adc value - X min offset
    tp_math = (tp_math * 239);          // X pixels 0-239
    tp_math = (tp_math / TP_CAL_XRANGE);
    tp_sx = tp_math;
    if(tp_sx > 239) tp_sx = 239;
  }
  else tp_sx = 0;

  // Convert Y adc value to screen pixels
  if(tp_y > TP_CAL_YMIN)            // make sure is >0 before doing calc!
  {
    tp_math = (tp_y - TP_CAL_YMIN);     // get Y adc value - Y min offset
    tp_math = (tp_math * 127);          // Y pixels 0-127
    tp_math = (tp_math / TP_CAL_YRANGE);
    tp_sy = tp_math;
    if(tp_sy > 127) tp_sy = 127;
  }
  else tp_sy = 0;
  tp_sy = (127 - tp_sy);    // invert Y to suit LCD
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b00000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;
  BacklightBlue   = 1;

  T0CON = 0b11000000;   // TMR0 enabled, 8bit, 1:1 prescaler

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(30);

  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 8);   // init for MikroC PRO version
  T6963C_init(240, 128, 8, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(0);         // cursor = off

  //-------------------------------------------------------
  // draw stuff on GLCD that remains constant.
  // write the text that will not change
  T6963C_Write_Text("SmartGLCD 3 Bargraph LED PWM", 0, 0, T6963C_ROM_MODE_OR);

  T6963C_Write_Text("Red   128", 0, 3, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("Green 128", 0, 6, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("Blue  128", 0, 9, T6963C_ROM_MODE_OR);

  T6963C_Write_Text("TX=",  0, 14, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("TY=",  0, 15, T6963C_ROM_MODE_OR);

  T6963C_Write_Text("SX=", 10, 14, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("SY=", 10, 15, T6963C_ROM_MODE_OR);

  //-------------------------------------------------------
  pwm_red   = 255;    // LED PWM starting values
  pwm_green = 64;
  pwm_blue  = 0;
  pwm_red_mem   = 128;  // LED PWM memorised starting values
  pwm_green_mem = 128;
  pwm_blue_mem  = 128;

  Draw_PWM_Value(0);    // draw all the objects
  Draw_PWM_Value(1);
  Draw_PWM_Value(2);
  Draw_Bargraph(0);     
  Draw_Bargraph(1);
  Draw_Bargraph(2);
  Draw_Button(0,0); 
  Draw_Button(1,0);

  Delay_ms(300);
  // turn TMR0 interrupt on (used for LED PWM)
  INTCON.GIE = 1;
  INTCON.TMR0IE = 1;

  // main running loop (repeat 10 times a second)
  while(1)
  {
    //---------------------------------------------
    // get TouchPanel X value
    TPfast_SetX();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_x = TPfast_Read();

    // get TouchPanel Y value
    TPfast_SetY();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_y = TPfast_Read();

    // format and display TP X value
    ByteToStr(tp_x,text);
    T6963C_Write_Text(text, 3, 14, T6963C_ROM_MODE_OR);

    // format and display TP Y value
    ByteToStr(tp_y,text);
    T6963C_Write_Text(text, 3, 15, T6963C_ROM_MODE_OR);

    //---------------------------------------------
    // get X Y coords as screen pixel coords; X=0-239 Y=0-127
    TP_convert_XY();  

    // format and display TP X value
    ByteToStr(tp_sx,text);
    T6963C_Write_Text(text, 13, 14, T6963C_ROM_MODE_OR);

    // format and display TP Y value
    ByteToStr(tp_sy,text);
    T6963C_Write_Text(text, 13, 15, T6963C_ROM_MODE_OR);

    //---------------------------------------------
    // if touched within total bargraph area, calc a 0-255 PWM value
    if(tp_sx>=(BAR_X-5) && tp_sx<=(BAR_X+BAR_LENGTH+5) &&
              tp_sy>=BAR_Y && tp_sy<=(BAR_Y+BAR_HEIGHT+24+24) )
    {
      // calc the pwm value 0-255 from pixel X position, put in tp_math
      if(tp_sx > BAR_X) 
      {
        tp_math = (tp_sx - BAR_X);   
        tp_math = (tp_math * 255); 
        tp_math = (tp_math / BAR_LENGTH);
        if(tp_math > 255) tp_math = 255;
      }
      else  tp_math = 0;
    
      // now check Y height, if within a bargraph update its pwm
      // Red Bargraph
      boff = 0;
      if(tp_sy>=(BAR_Y+boff) && tp_sy<=(BAR_Y+BAR_HEIGHT+boff) )
      {
        pwm_red = tp_math;
        Draw_Bargraph(0);
        Draw_PWM_Value(0);
      }
      // Green Bargraph
      boff = 24;
      if(tp_sy>=(BAR_Y+boff) && tp_sy<=(BAR_Y+BAR_HEIGHT+boff) )
      {
        pwm_green = tp_math;
        Draw_Bargraph(1);
        Draw_PWM_Value(1);
      }
      // Blue Bargraph
      boff = 24+24;
      if(tp_sy>=(BAR_Y+boff) && tp_sy<=(BAR_Y+BAR_HEIGHT+boff) )
      {
        pwm_blue = tp_math;
        Draw_Bargraph(2);
        Draw_PWM_Value(2);
      }
    }


    //---------------------------------------------
    // now test for the Playback and Memorise buttons!
    if(tp_sy > BUTY-2 && tp_sy < BUTY2+2)     // if Y is in button range
    {
      // test if Playback button touched
      if(tp_sx > 35 && tp_sx < 35+76)
      {
        Draw_Button(0,1);   // button 0, highlighted
        pwm_red   = pwm_red_mem;    // restore the pwm values
        pwm_green = pwm_green_mem;
        pwm_blue  = pwm_blue_mem;
        Draw_Bargraph(0);
        Draw_Bargraph(1);
        Draw_Bargraph(2);
        Draw_PWM_Value(0);
        Draw_PWM_Value(1);
        Draw_PWM_Value(2);
        // wait until button released
        while(1)    
        {
          TPfast_StartRead();     // start TP Y adc conversion
          tp_y = TPfast_Read();
          if(tp_y <= 12) break;   // if button was released
        }
        Draw_Button(0,0);   // button 0, normal
      }
      // test if Memorise button touched
      else if(tp_sx > 131 && tp_sx < 131+76)
      {
        Draw_Button(1,1);   // button 1, highlighted
        pwm_red_mem   = pwm_red;    // memorise the pwm values
        pwm_green_mem = pwm_green;
        pwm_blue_mem  = pwm_blue;
        while(1)    // wait until button released
        {
          TPfast_StartRead();     // start TP Y adc conversion
          tp_y = TPfast_Read();
          if(tp_y <= 12) break;   // if button was released
        }
        Draw_Button(1,0);   // button 1, normal
      }
    }
    //---------------------------------------------
    Delay_ms(100);
  }
}
//-----------------------------------------------------------------------------





