/******************************************************************************
    SG_Free_RAM.c     Using the free 32k RAM chip onboard the SmartGLCD!
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  The SmartGLCD has a 32 kbyte static RAM chip on-board, the RAM is used by
  the T6963C GLCD controller chip. As the GLCD only uses 6k to 9k of RAM
  that leaves 24+ kbytes of RAM for FREE that we can use to store anything!
  My own fnctions are used to write and read the RAM and my functions are
  very fast as they are optimised for SmartGLCD hardware.
  // NOTE! GLCD is in narrow text mode (6x8 font) giving 40x16 chars.
******************************************************************************/
// Global Variables
unsigned int  ram_add     absolute 0x15;    // overload vars for fast byte access
unsigned char ram_add_lo  absolute 0x15;
unsigned char ram_add_hi  absolute 0x16;

unsigned char i;          // used for loops

unsigned int etime;       // used to time events in uS

unsigned char txt[6];     // holds number for displaying on screen as text

unsigned char RSG_byte;   // used by my functions

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

//=============================================================================
//  RomanSG_CHECK_STATUS 
//=============================================================================
void RomanSG_Check_Status(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a very fast function to check if the
  // GLCD is busy. It returns when the GLCD is free.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  unsigned char read_val;
  TRISH = 0b11111111;   // set GLCD data port H to all inputs
  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F1 = 0;          // read is low

  while(1)              // loop here until GLCD is free
  {
    asm nop;     
    LATJ.F3 = 0;        // CE low
    asm nop;     
    read_val = PORTH;   // get the status byte
    LATJ.F3 = 1;        // CE high again
    if((read_val & 0x03) == 0x03) break;     // break if free
  }
}
//-----------------------------------------------------------------------------

//=============================================================================
//  RomanSG_READ_DATA
//=============================================================================
void RomanSG_Read_Data(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a very fast function to read one data byte
  // from the GLCD (from the RAM). The data byte will be put in
  // tho global variable RSG_byte.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  RomanSG_Check_Status();   // make sure GLCD is free first

  TRISH = 0b11111111;   // set GLCD data port H to all inputs

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ &= 0b11111100;   // rd and cd are made low
  asm nop;
  LATJ.F3 = 0;          // CE low
  asm nop;
  asm nop;
  RSG_byte = PORTH;      // read the data byte
  LATJ.F3 = 1;          // CE high again
  asm nop;

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F3 = 0;          // ce low (MikroE functions need this left low on exit!)
}
//-----------------------------------------------------------------------------

//=============================================================================
//  RomanSG_SEND_DATA
//=============================================================================
void RomanSG_Send_Data(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a very fast function to write one data byte
  // to the GLCD. The data byte is already in RSG_byte, for best speed.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  RomanSG_Check_Status();   // make sure GLCD is free first

  TRISH = 0b00000000;   // set GLCD data port H to all outputs
  LATH = RSG_byte;      // load the byte ready to the data port

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ &= 0b11111010;   // wr and cd are made low
  asm nop;
  LATJ.F3 = 0;          // CE low
  asm nop;
  LATJ.F3 = 1;          // CE high again
  asm nop;

  LATJ |= 0b00001111;   // ce wr rd cd
}
//-----------------------------------------------------------------------------

//=============================================================================
//  RomanSG_SEND_COMMAND
//=============================================================================
void RomanSG_Send_Command(void)
{
  //-------------------------------------------------------
  // Written for SmartGLCD module with T6963C GLCD only!
  // This is a test of a very fast function to write one command byte
  // to the GLCD. The command byte is already in RSG_byte, for best speed.
  // PORTJ = Res  CE   WR  RD  C/D  
  // PORTJ   4    3    2   1    0  
  //-------------------------------------------------------
  RomanSG_Check_Status();   // make sure GLCD is free first

  TRISH = 0b00000000;   // set GLCD data port H to all outputs
  LATH = RSG_byte;      // load the byte ready to the data port

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F2 = 0;          // write is low
  asm nop;
  LATJ.F3 = 0;          // CE low
  asm nop;
  LATJ.F3 = 1;          // CE high again
  asm nop;

  LATJ |= 0b00001111;   // ce wr rd cd
  LATJ.F3 = 0;          // ce low (MikroE functions need this left low on exit!)
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b01000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;     // green ON
  BacklightBlue   = 1;

  T1CON = 0b10110001;   // TMR1 on 16bit, 1:8 prescaler, used for time testing

  //-------------------------------------------------------
  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 6);   // init for MikroC PRO version
  T6963C_init(240, 128, 6, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(0);         // cursor = off

  Delay_mS(300);

  //-------------------------------------------------------
  // NOTE! GLCD is in narrow text mode (6x8 font) giving 40x16 chars.
  //-------------------------------------------------------
  // The code below might look a little messy! What it does is
  // increment the variable i then write that variable to ram
  // and read the variable back from ram, and display it.
  // this tests that all the static ram in the GLCD is available for our use!
  // 
  // The functions used to set the address for the ram, then write and read
  // the ram are my own fast functions. MikroC does not have a function
  // to read from the graphics ram. My functions are very fast and follow
  // the write/read protocols as specced in the T6963C GLCD datasheet.
  // 
  // In the SmartGLCD there is a 32kbyte ram chip (LY62256) and only
  // 8 kbytes or less is used for the GLCD so we have a FREE and FAST
  // static ram with 24 kbytes or more of storage! YAY!!
  //-------------------------------------------------------

  // write the top 2 lines to GLCD, it is text that will not change
  T6963C_Write_Text("Testing the FREE 24 kbytes of static RAM", 0, 0, T6963C_ROM_MODE_OR);  
  T6963C_Write_Text("that is on-board the SmartGLCD... YAY!!", 0, 1, T6963C_ROM_MODE_OR);  
  T6963C_Write_Text("RAM add.   Write  Read", 0, 3, T6963C_ROM_MODE_OR);  

  // Now loop and write different bytes to static ram in the free 24 kbyte
  // range between 8192-32767 (0x2000-0x7C17). 
  i = 0;
  ram_add = 8192;   // start of ram for "random" testing
  while(1)
  {
    //---------------------------------------------
    Delay_mS(1000);   // update new values every second
    
    //---------------------------------------------
    // test first address at 8192 (0x2000) AND test time this event using TMR1.
    TMR1H=0;
    TMR1L=0;              // clear TMR1 here
    RSG_byte = 0x00;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x20;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write a byte to ram, no change to address
    RomanSG_Send_Command();
    
    etime = TMR1L;        // read TMR1 here
    etime += (TMR1H << 8);
    WordToStr(etime,txt);    // get the time as text
    T6963C_Write_Text(txt, 22, 4, T6963C_ROM_MODE_OR);   // display time in uS
    T6963C_Write_Text("uS", 27, 4, T6963C_ROM_MODE_OR);  
  
    TMR1H=0;
    TMR1L=0;              // clear TMR1 here
    RSG_byte = 0x00;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x20;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();

    RSG_byte = 0xC5;        // command to read a byte from ram, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)

    etime = TMR1L;        // read TMR1 here
    etime += (TMR1H << 8);
    WordToStr(etime,txt);    // get the time as text
    T6963C_Write_Text(txt, 31, 4, T6963C_ROM_MODE_OR);   // display time in uS
    T6963C_Write_Text("uS", 36, 4, T6963C_ROM_MODE_OR);  
    
    // display the address to screen
    WordToStr(0x2000,txt);     
    T6963C_Write_Text(txt, 2, 5, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 5, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 5, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 5, T6963C_ROM_MODE_OR);
  
    //---------------------------------------------
    // test address at 12288 (0x3000)
    RSG_byte = 0x00;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x30;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write byte, no change to address
    RomanSG_Send_Command();
  
    RSG_byte = 0xC5;        // command to read a byte, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)
    
    // display the address to screen
    WordToStr(0x3000,txt);     
    T6963C_Write_Text(txt, 2, 7, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 7, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 7, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 7, T6963C_ROM_MODE_OR);
  
    //---------------------------------------------
    // test address at 16384 (0x4000)
    RSG_byte = 0x00;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x40;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write byte, no change to address
    RomanSG_Send_Command();
  
    RSG_byte = 0xC5;        // command to read a byte, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)
    
    // display the address to screen
    WordToStr(0x4000,txt);     
    T6963C_Write_Text(txt, 2, 9, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 9, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 9, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 9, T6963C_ROM_MODE_OR);
  
    //---------------------------------------------
    // test address at 20480 (0x5000)
    RSG_byte = 0x00;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x50;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write byte, no change to address
    RomanSG_Send_Command();
  
    RSG_byte = 0xC5;        // command to read a byte, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)
    
    // display the address to screen
    WordToStr(0x5000,txt);     
    T6963C_Write_Text(txt, 2, 11, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 11, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 11, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 11, T6963C_ROM_MODE_OR);
  
    //---------------------------------------------
    // test LAST address at 32767 (0x7FFF)
    RSG_byte = 0xFF;        // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = 0x7F;        // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write byte, no change to address
    RomanSG_Send_Command();
  
    RSG_byte = 0xC5;        // command to read a byte, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)
    
    // display the address to screen
    WordToStr(0x7FFF,txt);     
    T6963C_Write_Text(txt, 2, 13, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 13, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 13, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 13, T6963C_ROM_MODE_OR);
  
    //---------------------------------------------
    //---------------------------------------------
    // now test a "random" ram address, use a incrementing variable to
    // keep changing the address.
    ram_add += 727;       // add a prime number to address
    if(ram_add >= 32768)  // but keep it within legal 8192-31767 range
    {
      ram_add -= 24576;
    }
  
    RSG_byte = ram_add_lo;  // ram low address byte
    RomanSG_Send_Data();
    RSG_byte = ram_add_hi;  // ram hi address byte
    RomanSG_Send_Data();
    RSG_byte = 0x24;        // command to set ram address pointer
    RomanSG_Send_Command();
    
    i++;    // change i, it is a random value to write/read
    RSG_byte = i;           // the byte to write to ram
    RomanSG_Send_Data();
    RSG_byte = 0xC4;        // command to write byte, no change to address
    RomanSG_Send_Command();
  
    RSG_byte = 0xC5;        // command to read a byte, no change to address
    RomanSG_Send_Command();
    RomanSG_Read_Data();    // then read the data byte (into RSG_byte)
    
    // display the address to screen
    WordToStr(ram_add,txt);     
    T6963C_Write_Text("R", 0, 15, T6963C_ROM_MODE_OR);
    T6963C_Write_Text(txt, 2, 15, T6963C_ROM_MODE_OR);
    // display the write byte to screen
    ByteToStr(i,txt);      
    T6963C_Write_Text(txt, 12, 15, T6963C_ROM_MODE_OR);
    // display the read byte to screen
    ByteToStr(RSG_byte,txt);     
    T6963C_Write_Text(txt, 18, 15, T6963C_ROM_MODE_OR);
    // display a tag if write/read error occurs!
    if(i != RSG_byte) T6963C_Write_Text("ERROR!", 22, 15, T6963C_ROM_MODE_OR);
  }
}
//-----------------------------------------------------------------------------





