/******************************************************************************
     XTAL_Calib2.c    calibrates a PIC XTAL against GPS 1 second pulse
     Orig 9th May 2011 - Open source; www.RomanBlack.com
     
     The 1pps input is on PIC pin RC2, and period is captured using
     TMR1 CCP1 and a math process to derive MHz from the 2 16bit
     capture values while simply ignoring 16bit overflows.

     This is the same system as XTAL_Calib.c but has more math code
     to produce an accurate average for very fine xtal freq display.

     Update 10th May 2011; Instead of getting a timer reding each second
     and multiplying it by 4 to get xtal Hz, it now sums every 4 latest
     timer readings to get xtal Hz. This has 4 times less jitter and allows
     improved (faster) filtering while still being zero-error.

     The display shows;
     (line1) actual xtal frequency in Hz (new sample every second)
     (line2) actual xtal freq long-term running average
     
     MCU:             P16F887
     Dev.Board:       EasyPIC6
     Oscillator:      XT, 4 MHz xtal
     Ext. Modules:    LCD 2x16, GPS device with 1pps output pulse (SmartGPS)
     SW:              mikroC v7.0

******************************************************************************/
// global vars
#define XTAL_HZ 4000000    // user must enter this value to suit their PIC xtal
#define TIMER_HZ (XTAL_HZ / 4)
#define TIMER_MOD (TIMER_HZ % 65536)
unsigned int capture_last, capture_new, difference; // for capturing the xtal (timer) freq
signed int xerror;
unsigned long real_timer_freq;
signed long real_xtal_freq;
unsigned char txt[14];  // for text string display

#define FILT1 30    // faster filter used for first 60 seconds
#define FILT2 150   // slow filter used after that
unsigned int filter;
signed long ierror;
signed long real_xtal_av, real_xtal_avsub;
unsigned char cycles;   // how many samples were tested

unsigned long real_timer_buf[4];  // used to average every 4 timer readings
unsigned char rtb_count;

#include "RomanLCD.c"   // my 2x16 text LCD functions for EasyPIC6
//-----------------------------------------------------------------------------

//=============================================================================
//   MAIN 
//=============================================================================
void main()
{
	//----------------------------------------------------
	// setup the PIC 16F887
  ANSEL  = 0;             // Configure all PIC pins as digital
  ANSELH = 0;
	TRISB = 0b00000000; 	  // PORTB all outputs for LCD
	PORTC = 0;
  TRISC = 0b00000100; 	  // RC2 is connected to 1pps capture input
  TRISD = 0b11111111;
  
  // TMR1 (used to capture input period)
	T1CON =   0b00000001;  // TMR1 ON, 1:1 prescale
  CCP1CON = 0b00000101;  // CCP1, capture mode, every rising edge

	//----------------------------------------------------
  // setup LCD (using Romans LCD functions for EasyPIC6)
  RomanLCD_Init();
  RomanLCD_Out(0,0,"No Pulse");   // start message, until 1pps is detected

  // setup vars etc
  PIR1.CCP1IF = 0;   // clear capture flag
  cycles = 0;
  real_xtal_avsub = 0;
  filter = FILT1;   // initial avergaing filter value

  rtb_count = 0;    // initial setup for 4 count averaging buf
  real_timer_buf[0] = TIMER_HZ;
  real_timer_buf[1] = TIMER_HZ;
  real_timer_buf[2] = TIMER_HZ;
  real_timer_buf[3] = TIMER_HZ;
  
	//----------------------------------------------------
  // main run loop here. Every loop we measure the incoming 1pps signal
  // as a period and display the error and xtal Hz to the LCD.
  // NOTE! if there is no 1pps signal the PIC will appear to "hang"
  // until signal resumes.
  while(1)
  {
    //-----------------------------------------
    while(!PIR1.CCP1IF);                        // wait for 1pps input capture
    capture_new = CCPR1L + (CCPR1H * 256);      // get the 16bit capture 
    PIR1.CCP1IF = 0;                            // clear capture flag
    difference = (capture_new - capture_last);  // get diff between captures
    capture_last = capture_new;                 // store a capture copy for next time
    xerror = (difference - TIMER_MOD);          // get error compared to "perfect" xtal
    real_timer_freq = (TIMER_HZ + xerror);      // add error to timer

    // add the latest 4 timer readings together to get xtal Hz, this is
    // better with less jitter than the old system; timer*4 = Hz
    real_timer_buf[rtb_count] = real_timer_freq;  // put latest value in buffer
    rtb_count++;                                  // go to next buffer location
    rtb_count = (rtb_count & 0b00000011);         // only allow 0-3
    real_xtal_freq = (real_timer_buf[0] + real_timer_buf[1] + 
                      real_timer_buf[2] + real_timer_buf[3]);  // convert timer ticks to xtal Hz

    // format the xtal Hz value into a text string and display it
    LongToStr(real_xtal_freq,txt);    // format into text
    RomanLCD_Out(0,0,txt);            // display it on top line of LCD
    RomanLCD_Out(0,0,"Hz");

    //-----------------------------------------
    // the new sample is in; real_xtal_freq
    // calculate a running average in a Low Pass Filter accumulator.
    // only do averaging after it has settled down (more than 4 captures)
    if(cycles < 255)  cycles++;        // will stop counting at 255
    if(cycles == 5)  real_xtal_av = real_xtal_freq;   // starting value for average
    if(cycles > 5)   
    {
      // Now do the averaging. The process is to average the small integer error 
      // between the captured value and the integer portion of the average.
      // these small integer errors are then averaged in real_xtal_avsub,
      // and this becomes extra resoluton data (sub-integer data) of the average.
      // the sub-integer data is kept as decimal, scaled so; 1000000 = 1 integer
      if(cycles == 65)
      {
        filter = FILT2;             // use finer filtering after first 60 good samples
        RomanLCD_Out(0,15,"f");     // show "fine" tag on LCD
      }
      ierror = (real_xtal_freq - real_xtal_av); // get small integer error
      real_xtal_avsub -= ((real_xtal_avsub+(filter/2))/filter);   // remove a rounded % of average
      ierror = (ierror * (1000000/filter));     // scale new error to % of filter value
      real_xtal_avsub += ierror;                // add into sub-integer average

      // if at any point the sub-integer average becomes >1 or <0 we remove
      // integer counts from it and put them in the integer average.
      while(real_xtal_avsub >= 1000000)    // if sub is > 1 integer
      {
        real_xtal_avsub -= 1000000;
        real_xtal_av += 1;
      }
      while(real_xtal_avsub < 0)           // if sub is < 0 integer
      {
        real_xtal_avsub += 1000000;
        real_xtal_av -= 1;
      }

      // at this point we must have an integer average in real_xtal_av
      // and the 0-0.999999 sub-integer resolution in real_xtal_avsub 
            
      // display the 0-0.999999 data that will appear after the decimal point
      // this is currently in a 6 digit format; 0-999999
      ierror = (real_xtal_avsub / 1000);  // make 6 dec digits into 3 decimal digits
      LongToStr(ierror,txt);              // format into text string
      txt[7] = '.';                       // manually insert dec point
      if(txt[8] == ' ') txt[8] = '0';     // add leading zeros where needed
      if(txt[9] == ' ') txt[9] = '0'; 
      RomanLCD_Out(1,4,txt);              // display it
    
      // and lastly display the integer average
      LongToStr(real_xtal_av,txt);      // format into text string
      RomanLCD_Out(1,0,txt);            // display it
      RomanLCD_Out(1,0,"av");
    }
  }  
}
//-----------------------------------------------------------------------------

// tests;
// 4MHz xtal; 24.0'C = 4000166.71 Hz (41.7 PPM fast)
// 8MHz xtal; 24.0'C = 7999830.20 Hz (21.2 PPM slow)


