/******************************************************************************
  SH1_Charge.c   Battery solar charge controller
  Open-source  -  2nd Nov 2009  -  Roman Black
    www.RomanBlack.com/shift1/sh1_projects.htm

  This displays the battery voltage, battery volts is measured on AN0.
  It also acts as a charge controller to turn something on (or off)
  when battery voltage > setpoint. Setpoint is adjustable and displayed
  on the LCD, and is saved to eeprom so is maintained even if power cut.
  Also pressing both adjust buttons changes the voltage display range;
  there are 4 ranges suitable to display all battery voltages
  ie 6v 12v 24v 36v 48v etc. 
  
  Note! After battery > setpoint load turns ON, and will remain ON
  for a minimum time of 10 seconds.
  
  Note! Use a 10k pot for the adc voltage to GP0, this sets the
  voltage display calibration, also use a large cap (470uF?)
  from GP0 to ground to smooth voltage and remove noise.

  PIC pins;
   GP0 = ADC in (battery volts, from a 10k pot which sets calibration)
   GP1 = button (lo = pressed) adjust Setpoint UP
   GP2 = digital output to Shift1-LCD
   GP3 = button (lo = pressed) adjust Setpoint DOWN (both pressed; change range)
   GP4 = load control output; HI when battery > setpoint
   GP5 = load control output (inverted); LO when battery > setpoint

******************************************************************************/

// global variables;

unsigned int adc_volts absolute 0x20;     // ADC "voltage" 10bit value
unsigned char adc_voltsL absolute 0x20;   // and allow access to each byte
unsigned char adc_voltsH absolute 0x21;   // by overloading variables

unsigned int set_volts absolute 0x22;     // setpoint voltage 
unsigned char set_voltsL absolute 0x22;   
unsigned char set_voltsH absolute 0x23;

unsigned char buttons;         // buffer for buttons
unsigned char adj_res;         // adjust button change resolution
unsigned char range;           // voltage display range
unsigned char pulse;           // for timing the display pulse indicator
unsigned char load_timer;      // for timing a minimum load ON period

char txt[8];             // used to display number string

#define EE_RANGE  0x00    // EEPROM addresses for stored data
#define EE_SETL   0x01    // setpoint lo byte
#define EE_SETH   0x02    // setpoint hi byte

// function declarations;
void message1(void);      // text messages to draw to LCD
void message2(void);

//=============================================================================


//-----------------------------------------------------------------------------
// this line adds the Shift1 and LCD functions
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


// wrap delay as a function call to save code size
void delay_500ms(void)
{
  Delay_mS(460);  // this value gives about 1 second per "pulse"
}


//=============================================================================
//  TEST VOLTAGE
//=============================================================================
void test_voltage(void)
{
  //-------------------------------------------------------
  // read battery voltage from AN0 (RA0)
  ADCON0.GO = 1;      // start ADC conversion
  while(ADCON0.GO);   // wait for conversion to be finished

  adc_voltsL = ADRESL;  // save bottom 8 bits
  adc_voltsH = ADRESH;  // and top 2 bits
}
//-----------------------------------------------------------------------------

//=============================================================================
//  FORMAT VOLTAGE
//=============================================================================
void format_voltage(unsigned int volts)
{
  //-------------------------------------------------------
  // correct the battery voltage for range;
  // range0. 6v note! ADC 1000 = 10.00v,   (res = 0.01v)
  // range1. 12v note! ADC 1000 = 20.00v,  (res = 0.02v) ADC 600 = 12.00v 
  // range2. 24v note! ADC 1000 = 50.00v,  (res = 0.05v)
  // range2. 36v note! ADC 1000 = 50.00v,  (res = 0.05v)
  // range3. 48v note! ADC 1000 = 100.00v, (res = 0.10v)
  
  if(range == 1) volts = (volts * 2);
  if(range == 2) volts = (volts * 5);
  if(range == 3) volts = (volts * 10);
  
  //---------------------------------------------
  // format voltage int to text
  wordtostr(volts,txt);
  txt[5] = txt[4];    // move 2 decimal places
  txt[4] = txt[3];    
  txt[3] = '.';       // add dec point
  txt[6] = 'v';       // add volts
  txt[7] = 0;         // and add NULL
}
//-----------------------------------------------------------------------------


//=============================================================================
//   MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // 12F675 setup registers etc
  
  ANSEL =  0b00100001;   // ADC fosc32, AN0 is analog input
  ADCON0 = 0b10000001;   // right justified, VREF=Vdd, ADC=AN0, ADC is ON

  CMCON = 0x07;   // comparators OFF

  TRISIO = 0b00001011;    // GP2 out, GP1,3 buttons, GP0 is ADC in
  GPIO =   0b00100100;    // GP2 normally HI, goes to Shift1-LCD, GP5 norm HI
  WPU =    0b00000010;    // pin pullups; 1 = pullup ON


  //-------------------------------------------------------
  // setup timers
  // TMR0 is a free running timer, at 1:1 (8MHz xtal = 2MHz)
  OPTION_REG = 0b00001000;  // pin pullups enabled, TMR0 = 1:1 prescale

  //-------------------------------------------------------
  // small delay for PSU to stabilise
  delay_500ms();

  // get variables from PIC internal eeprom
  range = eeprom_read(EE_RANGE);  // get range from eeprom stored value
  if(range == 0xFF)               // if eeprom is blank, insert default values
  {
    range = 1;                    // default range is range1 (12v)
    eeprom_write(EE_RANGE,range);
    set_voltsH = 0x02;            // default setvolts is 0x02B2, 690 (13.80v)
    set_voltsL = 0xB2;
    eeprom_write(EE_SETL,set_voltsL);
    eeprom_write(EE_SETH,set_voltsH);
  }
  else    // else if eeprom range was good,  get the other values from eeprom 
  {
    set_voltsL = eeprom_read(EE_SETL);  
    set_voltsH = eeprom_read(EE_SETH);  
  }

  // setup any other variables
  load_timer = 0;
  adj_res = 0;

  //-------------------------------------------------------
  // initialise the LCD
  SH1_Lcd_Init();
  SH1_lcd_backlight = 0;  // LCD backlight is OFF

  // show startup message
  SH1_Lcd_Move(0,0); // start of line0
  message1();     // "Battery"
  message2();     // "Controller v1.0"

  delay_500ms();  // 1 second delay
  delay_500ms();

  // now setup the main operating display
	SH1_Lcd_Byte(0x01,SCMD);    // clear display command (again)
  SH1_Lcd_Move(0,2); 
  message1();     // "Battery"

  SH1_Lcd_Move(1,5); 
  SH1_Lcd_Char('S');   // "SetP"
  SH1_Lcd_Char('e');
  SH1_Lcd_Char('t');
  SH1_Lcd_Char('P');

  //-------------------------------------------------------
  // TEMP!
  // display the setpoint volts
  format_voltage(set_volts);  // format by range, add dec point etc
  SH1_Lcd_Out(1,9,txt);       // display it


  //-------------------------------------------------------
  // now main run loop here
  while(1)
  {
    //---------------------------------------------
    // loop executes roughly every 0.5 sec
    delay_500ms();

    // read the buttons
    buttons = GPIO;
    buttons &= 0b00001010;  // only keep the 2 actual buttons

    // handle buttons if they are pressed (LO = pressed)
    if(buttons < 0b00001010)
    {
      if(!buttons)  // if both buttons pressed!
      {
        // change voltage display range
        range++;
        if(range > 3) range = 0;
        eeprom_write(EE_RANGE,range); // save value to eeprom too!
      
        // must display the voltages again
        format_voltage(adc_volts);  // format by range, add dec point etc
        SH1_Lcd_Out(0,9,txt);       // display it
        
        format_voltage(set_volts);  // format by range, add dec point etc
        SH1_Lcd_Out(1,9,txt);       // display it
      
      }
      else
      {
        if(!GPIO.F1)    // setpoint UP button pressed
        {
          if(!adj_res)
          {
            if(set_volts < 1023) set_volts++;
          }
          else
          {
            if(set_volts <= 1018) set_volts+=5;
          }
          adj_res = 1;
          format_voltage(set_volts);  // format by range, add dec point etc
          SH1_Lcd_Out(1,9,txt);       // display it
          eeprom_write(EE_SETL,set_voltsL); // save value to eeprom too!
          eeprom_write(EE_SETH,set_voltsH);
        }
        if(!GPIO.F3)    // setpoint DOWN button pressed
        {
          if(!adj_res)
          {
            if(set_volts) set_volts--;
          }
          else
          {
            if(set_volts >= 5) set_volts-=5;
          }
          adj_res = 1;
          format_voltage(set_volts);  // format by range, add dec point etc
          SH1_Lcd_Out(1,9,txt);       // display it
          eeprom_write(EE_SETL,set_voltsL); // save value to eeprom too!
          eeprom_write(EE_SETH,set_voltsH);
        }  
      }
    }
    else
    {
      adj_res = 0;
    }

    //---------------------------------------------
    // display the pulse to show unit is operating
    pulse++;
    SH1_Lcd_Move(0,0); 
    if(pulse.F0) SH1_Lcd_Char('*');
    else         SH1_Lcd_Char(' ');
    
    // read and display battery voltage
    test_voltage();             // read volts from adc
    format_voltage(adc_volts);  // format by range, add dec point etc
    SH1_Lcd_Out(0,9,txt);       // display it

    // now control the load and display LOAD message
    if(adc_volts >= set_volts)  load_timer = 20;
    
    // display "LOAD" message on LCD if load is ON
    if(load_timer)
    {
      GPIO.F4 = 1;        // actual LOAD ON
      GPIO.F5 = 0;

      SH1_Lcd_Move(1,0);  // LOAD message
      SH1_Lcd_Char('L');
      SH1_Lcd_Char('O');
      SH1_Lcd_Char('A');
      SH1_Lcd_Char('D');
    }
    else
    {
      GPIO.F4 = 0;        // actual LOAD OFF
      GPIO.F5 = 1;

      SH1_Lcd_Move(1,0);  // clear LOAD message
      SH1_Lcd_Char(' ');
      SH1_Lcd_Char(' ');
      SH1_Lcd_Char(' ');
      SH1_Lcd_Char(' ');
    }      
    if(load_timer) load_timer--; // gives a load ON minimum period
    
    //---------------------------------------------
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MESSAGES     LCD text messages, done like this to save some RAM 
//=============================================================================
void message1(void)
{
  SH1_Lcd_Char('B');
  SH1_Lcd_Char('a');
  SH1_Lcd_Char('t');
  SH1_Lcd_Char('t');
  SH1_Lcd_Char('e');
  SH1_Lcd_Char('r');
  SH1_Lcd_Char('y');
}
//-------------------------------------------------------
void message2(void)
{
  SH1_Lcd_Move(1,0); // start of line1
  SH1_Lcd_Char('C');
  SH1_Lcd_Char('o');
  SH1_Lcd_Char('n');
  SH1_Lcd_Char('t');
  SH1_Lcd_Char('r');
  SH1_Lcd_Char('o');
  SH1_Lcd_Char('l');
  SH1_Lcd_Char('l');
  SH1_Lcd_Char('e');
  SH1_Lcd_Char('r');
  SH1_Lcd_Char(' ');
  SH1_Lcd_Char('v');
  SH1_Lcd_Char('1');
  SH1_Lcd_Char('.');
  SH1_Lcd_Char('0');
}
//-----------------------------------------------------------------------------



