/******************************************************************************
  SH1_Tacho.c   A tacho (RPM meter) for rotating machinery.
  (requires;) #include "Shift1_LCD.c"
  Open-source  -  2nd Nov 2009  -  www.RomanBlack.com

  This measures RPM by getting a TMR1 16bit period for each
  input pulse; done 4 times for a total of 4 input pulses. 
  This total period represents the period of 1 pulse in actual
  microseconds.
  So RPM is calculated by; RPM = (60000000 / period)

  The display shows RPM from one test (4 pulses) and it
  also displays an average RPM from 4 tests (16 pulses).
  This is useful for moving machinery where the speed is
  varying due to cutting tasks, the average RPM is a better
  indication of real machine RPM.

******************************************************************************/

// global vars
unsigned int period absolute 0x20;
unsigned char period0 absolute 0x20;
unsigned char period1 absolute 0x21;

#define PIN_P0   GPIO.F0    // these 2 pins select; pulses per rev
#define PIN_P1   GPIO.F1    //
#define PIN_RPM  GPIO.F3    // the input signal

unsigned char pulsecount;   // used to test 4 input pulses
unsigned char avcount;      // used to average 4 RPM values

unsigned int rpm;           // actual RPM value to display
unsigned int rpm_avg[4];    // last 4 RPM values, used to average them

unsigned long accum;        // these used for 32bit math to calc RPM
unsigned long math;

unsigned char txt[6];       // used to hold text string to display 5digit RPM

//-----------------------------------------------------------------------------
// this line adds my routines for Shift1 LCD driving
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


//=============================================================================
//   TEST RPM
//=============================================================================
void test_rpm(void)
{
  //-----------------------------------------------------
  // this waits for one full LOW period of the RPM input
  // signal, then times the period from / to / edge of the
  // input signal. This gives a period of one full input pulse.
  // TMR1 is running at 1:8 prescaler (250kHz), this means
  // a very long pulse will cause a timeout error.
  //
  // On exit; period contains the 16bit period.
  // NOTE! if timeout error occurs, period will return zero
  //-----------------------------------------------------

  // clear vars first
  period = 0;
  TMR1L = 0;
  TMR1H = 0;

  //-----------------------------------------------------
  // wait for full pulse first to sync pulse edges
  while(!PIN_RPM);  // wait for / edge
  while(PIN_RPM);   // wait for \ edge
  
  //-----------------------------------------------------
  // now we are synced, start TMR1 on next / edge
  while(!PIN_RPM);  // wait for / edge
  T1CON = 0b00110001;   // TMR1 1:8 prescale, TMR1 ON

  while(PIN_RPM)   // wait for \ edge
  {
    if(TMR1H == 255) return;
  }

  // now stop TMR1 on next / edge
  while(!PIN_RPM)  // wait for / edge
  {
    if(TMR1H == 255) return;
  }
  T1CON = 0b00110000;   // TMR1 1:8 prescale, TMR1 OFF

  //-----------------------------------------------------
  // now we have captured the TMR1 value for one pulse period
  // put that 16bit TMR1 value in period
  period0 = TMR1L;
  period1 = TMR1H;

  // wait for low again before exit
  while(PIN_RPM);   // wait for \ edge

}
//-----------------------------------------------------------------------------


// wrap this delay in a function to save ROM
void Delay_mS_400(void)
{
   Delay_mS(400);
}


//=============================================================================
//   MAIN
//=============================================================================
void main ()
{
  //-----------------------------------------------------
  // PIC 12F675  setup ports

  ANSEL = 0;
  CMCON = 0x07;

  TRISIO = 0b00001011;  // GP2 is Shift1 out, GP1,2 inputs set range, GP3 is tacho input
  GPIO =   0b00000100;  // GP2 high default for Shift1 out
  WPU =    0b00000011;  // pin pullups; 1 = pullup ON

  //-----------------------------------------------------
  // setup timers
  
  OPTION_REG = 0b00000000; // pin pullups enabled
  
  //T1CON = 0b00110001;   // TMR1 1:8 prescale, TMR1 ON
  T1CON = 0b00110000;   // TMR1 1:8 prescale, TMR1 OFF
  
  // setup vars
  avcount = 0;

  rpm_avg[0] = 0;   // clear average before start
  rpm_avg[1] = 0;
  rpm_avg[2] = 0;
  rpm_avg[3] = 0;

  //-----------------------------------------------------
  // initialise LCD
  Delay_mS_400();
  SH1_Lcd_Init();           // init LCD
  SH1_lcd_backlight = 0;    // Shift1-LCD backlight OFF

  SH1_Lcd_Out(0,0,"Tacho");   // start message (optional)
  Delay_mS_400();
  
  //-----------------------------------------------------

  // main run loop here
  while(1)
  {
    //-------------------------------------
    // small delay of about 100mS per loop
    Delay_mS(100);

    // now test the RPM frequency over 4 input pulses
    rpm = 0;
    pulsecount = 4;
    math = 2;     // add half the divisor to stop rounding errors
    while(pulsecount)
    {
      test_rpm();
      if(!period) goto display_rpm;  // catch timeout error (rpm too low)
      math += period;
      pulsecount--;
    }

    //-------------------------------------
    // at this point, math contains a total period for 4 pulses
    // now we need to convert that to RPM.
    // there are 4 ranges, which depend on the machine hardware
    // these are selected by PIC pins PIN_P1 and PIN_P0;
    //  11 = 1 pulse per lathe revolution; RPM = 60000000 / period
    //  10 = 2 pulse per lathe revolution; RPM = 30000000 / period
    //  01 = 4 pulse per lathe revolution; RPM = 15000000 / period
    //  00 = 8 pulse per lathe revolution; RPM =  7500000 / period

    if(PIN_P1)
    {
      if(PIN_P0) accum = 60000000;  // 11; 1 pulse per lathe revolution
      else       accum = 30000000;  // 10; 2 pulse per lathe revolution
    }
    else  
    {
      if(PIN_P0) accum = 15000000;  // 01; 4 pulse per lathe revolution
      else       accum =  7500000;  // 00; 8 pulse per lathe revolution
    }
    
    // (note! this divide system uses 481 less ROM than compiler div32x32)
    // (note2! add half the divisor before divide, to fix rounding down error)
    accum += (math / 2);  // add half the divisor to the total
    rpm = 0;
    while(accum >= math)  // this is; rpm = (accum / math);
    {
      accum -= math;
      if(rpm < 65000) rpm++;  // fix at 65000 max RPM displayed
    }

    //-------------------------------------
    display_rpm:
    // format RPM to text, then display it on LCD
    
    wordtostr(rpm,txt);     // insigned int to text
    SH1_Lcd_Out(0,0,txt);   // display it

    SH1_Lcd_Move(0,6);      // display "RPM"
    SH1_Lcd_Char('R');
    SH1_Lcd_Char('P');
    SH1_Lcd_Char('M');

    //-------------------------------------
    // now calculate an average RPM from the last 4 RPM values
    
    rpm_avg[avcount] = rpm;   // include the new RPM value
    avcount++;
    if(avcount >= 4) avcount = 0;

    // now add the 4 last RPM values
    math = (rpm_avg[0] + rpm_avg[1] + rpm_avg[2] + rpm_avg[3]);
    math += 2;    // add half the divisor to stop rounding errors

    // divide the total by 4 to give an average
    math = (math / 4);
    rpm = math;

    // now display the average
    wordtostr(rpm,txt);
    SH1_Lcd_Out(0,10,txt);

    SH1_Lcd_Move(0,15);
    SH1_Lcd_Char(0xA5); // pick a non-distracting char for average
  }
}
//-----------------------------------------------------------------------------













