/******************************************************************************
  SH1_Temp2.c   Temperature controller using LM335 temp sensor
  Open-source  -  5th Nov 2009  -  Roman Black
    www.RomanBlack.com/shift1/sh1_projects.htm

  This is a complete temperature controller. The temperature
  control "Setpoint" can be set anywhere from -40'C to +100'C.
  (This is the operating range of the LM335 temp sensor)
  It can be used for incubators, room heaters, freezers etc.
  This was based on SH1_Temp.c but has been improved with a
  1000 hour timer added, and timer reset button.

  Two buttons control the setpoint temperature; UP and DOWN
  Hold these to adjust the setpoint.
  
  The 16x2 LCD shows;
  (top line) the 000:00:00 timer, and HEAT / COOL indicator
  (bottom line) -xxx.x'C (actual temp) -xxx.x'C (setpoint temp)
  
  PIC pins;
    GP0 = ADC in, connects to Vout pin of LM335 temperature sensor
    GP1 = button (lo = pressed) adjust Setpoint UP
    GP2 = digital output to Shift1-LCD
    GP3 = button (lo = pressed) adjust Setpoint DOWN (needs 10k pullup resistor!)
    GP4 = HEAT control output; HI when temp < setpoint
    GP5 = button - timer reset button

******************************************************************************/

// global variables;

signed int set_temp absolute 0x20;       // setpoint voltage 
unsigned char set_tempL absolute 0x20;   // overload vars for smaller code
unsigned char set_tempH absolute 0x21;

signed int adc_temp;           // ADC "voltage" 10bit value

unsigned int utemp;            // used in temp calcs

unsigned char buttons;         // buffer for buttons
unsigned char adj_res;         // adjust button change resolution
unsigned char load_timer;      // for timing a minimum load ON period

char txt[9];                    // used to display number string

#define EE_SETL   0x00          // EEPROM addresses for stored data;
#define EE_SETH   0x01          // (setpoint lo and hi bytes)

unsigned int bres;        // used for 1 second bresenham timer
unsigned int hours;       // timer clock values
unsigned char mins;       //
unsigned char secs;       //

//=============================================================================


//-----------------------------------------------------------------------------
// this line adds the Shift1 and LCD functions
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


// wrap delay as a function call to save code size
void delay_500ms(void)
{
  Delay_mS(460);  // this value gives about 1 second per "pulse"
}



//=============================================================================
//   UPDATE CLOCK
//=============================================================================
void update_clock(void)
{
  //-----------------------------------------------------
  // this updates the clock values every new second
  // note! secs is incremented in main()
  if(secs >= 60)
  {
    secs = 0;
    mins++;  
  }  
  if(mins >= 60)
  {
    mins = 0;
    hours++;  
  }  
  if(hours > 999)  
  {
    hours--;      // fix at 999, is better than resetting to 0
  }  
}
//-----------------------------------------------------------------------------

//=============================================================================
//   DISPLAY CLOCK
//=============================================================================
void display_clock(void)
{
  //-----------------------------------------------------
  // display the clock values to the LCD
    
  // move to home position
  
  // show hours 0-999 (is a 1000 hour timer)
  wordtostr(hours,txt);
  txt[5] = ':';
  txt[6] = 0;
  SH1_Lcd_Out(0,0,txt+2);

  // show mins
  SH1_Lcd_Move(0,4);  
  SH1_Lcd_Char('0' + (mins / 10));
  SH1_Lcd_Char('0' + (mins % 10));
  SH1_Lcd_Char(':');

  // show secs
  SH1_Lcd_Char('0' + (secs / 10));
  SH1_Lcd_Char('0' + (secs % 10));
}
//-----------------------------------------------------------------------------

//=============================================================================
//  TEST TEMPERATURE
//=============================================================================
void test_temperature(void)
{
  //-------------------------------------------------------
  // test temperature procedure;
  //  1. read the ADC voltage from the LM335 temp sensor
  //  2. convert the ADC reading into degrees Kelvin
  //  3. subtract a calibration offset, this turns 'K into 'C
  //-------------------------------------------------------

  // read voltage from AN0 (RA0) which is the LM335 temperature sensor.
  ADCON0.GO = 1;      // start ADC conversion
  while(ADCON0.GO);   // wait for conversion to be finished

  utemp = ADRESH;      // put the 10 bit ADC value in signed 16bit var
  utemp = (utemp * 256);
  utemp += ADRESL;  

  // convert the ADC reading into 'K
  // each ADC count is = 0.5' so multiply by 5, this gives 'K *10
  // (so it adds one decimal place)
  utemp = (utemp << 2) + utemp;  // adc_temp = (adc_temp * 5)
  adc_temp = utemp;       // copy to signed var now
  
  // now convert from 'K to 'C by subtracting a cailbration
  // value. With a perfect LM335 and 5.00v supply, the value
  // at 25'C is; 1024/5.00v*2.98v = 610 ADC counts = (x5) 3050 
  // which needs to be 25'C or 250, so (3050-250) = 2800 cal value.
  // My 5v supply is actual 5.07v (fairly typical) so thats;
  // 1024/5.07*2.98 = 602 ADC = (x5) 3010; (3010-250) = 2760 cal value.

  // note! you can also adjust this (5 counts = 0.5'C)  to "trim"
  // the LM335. Just test displayed temp against a proper thermometer.
  // My test LM335 reads about 0.4'C low, so; (2760-5) = 2755
  adc_temp -= 2755;    // subtract the calibration value
}
//-----------------------------------------------------------------------------


//=============================================================================
//  FORMAT TEMPERATURE
//=============================================================================
void format_temperature(signed int ftemp)
{
  //-------------------------------------------------------
  // formats the temperature value into a text string
  // so it can be diplayed on LCD.
  // initially temp is a number; [- nnnn]
  // we format that to show;     [-nnn.n'C]
  //-------------------------------------------------------

  // convert 4 digit number to text string
  inttostr(ftemp,txt);   

  // format the data within the text string
  txt[1] = txt[2];    // move 3 digits, 1 place to the left
  txt[2] = txt[3];  
  txt[3] = txt[4];  
  txt[4] = '.';       // add dec point
  txt[6] = 0xDF;      // add 'C chars
  txt[7] = 'C';       
  txt[8] = 0;         // alaways add NULL

  // and correct for 0.x and -0.x situations
  if(txt[3]==' ') txt[3] = '0';  // change  .n to 0.n
  if(txt[3]=='-')                // change -.n to -0.n
  {
    txt[2] = '-'; 
    txt[3] = '0'; 
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//   MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // 12F675 setup registers etc
  
  ANSEL =  0b00100001;   // ADC fosc32, AN0 is analog input
  ADCON0 = 0b10000001;   // right justified, VREF=Vdd, ADC=AN0, ADC is ON

  CMCON = 0x07;   // comparators OFF

  TRISIO = 0b00101011;    // GP2 out, GP1,3,5 buttons, GP0 is ADC in
  GPIO =   0b00000100;    // GP2 normally HI, goes to Shift1-LCD
  WPU =    0b00100010;    // pin pullups; 1 = pullup ON

  //-------------------------------------------------------
  // setup timers
  // TMR0 not used
  OPTION_REG = 0b00001000;  // pin pullups enabled, TMR0 = 1:1 prescale

  // TMR1 used for 1 second "zero-error" timer
  T1CON = 0b00100001;    // TMR1 1:4 prescale  = 250kHz (roll = 3.8 Hz)

  //-------------------------------------------------------
  // small delay for PSU to stabilise
  delay_500ms();

  // get the previous setpoint variables from PIC internal eeprom
  set_tempL = eeprom_read(EE_SETL);  
  set_tempH = eeprom_read(EE_SETH);  

  // check and fix invalid set_temp (like if eeprom was blank!)
  if(set_tempH == 0xFF)  
  {
    set_temp = 400;                  // use default set_temp of +40'C
    eeprom_write(EE_SETL,set_tempL); // and save +40'C to eeprom too
    eeprom_write(EE_SETH,set_tempH);
  }

  // setup any other variables
  load_timer = 0;
  adj_res = 0;
  bres = 0;
  secs = 0;
  mins = 0;
  hours = 0;
  
  //-------------------------------------------------------
  // initialise the LCD
  SH1_Lcd_Init();
  SH1_lcd_backlight = 0;  // LCD backlight is OFF

  delay_500ms();
  delay_500ms();

  //-------------------------------------------------------
  // TEMP!
  // display the setpoint volts
  format_temperature(set_temp);   // format by range, add dec point etc
  SH1_Lcd_Out(1,8,txt);           // display it

  //-------------------------------------------------------
  // now main run loop here
  while(1)
  {
    //---------------------------------------------
    // wait here for one roll of TMR1 (this is about 0.26 seconds)
    // this 1/4 second is used for button updating

    while(!PIR1.TMR1IF);    // wait here for TMR1 roll flag
    PIR1.TMR1IF = 0;        // now clear it
    
    //---------------------------------------------
    // test the 2 buttons and adjust the temperature setpoint
    // temperature setpoint is constrained by the
    // LM335 temperature limts; -40'C to +100'C
    
    if(!GPIO.F1)    // setpoint UP button pressed
    {
      if(!adj_res)
      {
        if(set_temp < 1000-5) set_temp+=5;   // 100'C  step = 0.5'C
      }
      else
      {
        if(set_temp < 1000-25) set_temp+=25; // 100'C  step = 2.5'C
      }
      adj_res = 1;                      // speed buttons up
      format_temperature(set_temp); 
      SH1_Lcd_Out(1,8,txt);             // display new settemp
      eeprom_write(EE_SETL,set_tempL);  // save value to eeprom too!
      eeprom_write(EE_SETH,set_tempH);
    }
    if(!GPIO.F3)    // setpoint DOWN button pressed
    {
      if(!adj_res)
      {
        if(set_temp >= -40 +5)  set_temp -= 5;  // -40'C step = 0.5'C
      }
      else
      {
        if(set_temp >= -40 +25) set_temp -= 25; // -40'C step = 2.5'C
      }
      adj_res = 1;                      // speed buttons up
      format_temperature(set_temp); 
      SH1_Lcd_Out(1,8,txt);             // display new settemp
      eeprom_write(EE_SETL,set_tempL);  // save value to eeprom too!
      eeprom_write(EE_SETH,set_tempH);
    }  
    // slow the buttons again once both are released
    if(GPIO.F1 && GPIO.F3) adj_res = 0;

    //---------------------------------------------
    // now check if one second has passed, this uses my  "zero error" 
    // 1 second timing system, see; www.RomanBlack.com/one_sec.htm
    // 250000 TMR1 counts = 1 second.
    //
    // Tweak this value to suit this PIC RCosc !
    // calibrate this value by testing your PIC over 1 hour (3600 seconds)
    // and measure the error; ie if your timer reads 7 seconds fast;
    // 250000 / 3600 * (3600 + 7) = 250486

    #define TICKSECOND 250000     // (tweak this value to suit each PIC!)

    bres += (65536/8);    // add 1 TMR1 roll period to total

    if(bres < (TICKSECOND/8))    // if < 1 second
    {
      continue;   // not 1 second yet! so go back to start
    }

    // gets to here after one second!
    bres -= (TICKSECOND/8);   // so subtract exactly 1 second, retain error

    //---------------------------------------------
    // now update the 1 second clock timer
    secs++;                // add a second to timer
    if(!GPIO.F5)           // if clear timer button is pressed;
    {
      secs = 0;            // clear the timer
      mins = 0;
      hours = 0;
    }
    update_clock();        // update and display the timer
    display_clock();

    //---------------------------------------------
    // also read and display temperature once per second
    test_temperature();             // read volts from adc
    format_temperature(adc_temp);   // format by range, add dec point etc
    SH1_Lcd_Out(1,0,txt);           // display it

    //---------------------------------------------
    // now control the HEAT and display HEAT message
    // load_timer is set to 3 which means the heater element
    // must turn on for a minimum of 3 seconds.
    if(adc_temp < set_temp)  load_timer = 3;
    
    // display "HEAT" message on LCD if heat is ON
    if(load_timer)
    {
      GPIO.F4 = 1;        // actual HEAT ON
      GPIO.F5 = 0;

      SH1_Lcd_Move(0,12);  // HEAT message
      SH1_Lcd_Char('H');
      SH1_Lcd_Char('E');
      SH1_Lcd_Char('A');
      SH1_Lcd_Char('T');
      load_timer--; // gives a load ON minimum period
    }
    else     // else is COOL
    {
      GPIO.F4 = 0;        // actual HEAT OFF
      GPIO.F5 = 1;

      SH1_Lcd_Move(0,12);  // clear HEAT message
      SH1_Lcd_Char('C');
      SH1_Lcd_Char('O');
      SH1_Lcd_Char('O');
      SH1_Lcd_Char('L');
    }      
    //---------------------------------------------
  }
}
//-----------------------------------------------------------------------------




